from isf_db_helper import DbHelper

import sqlite3

class SushiModel:
    """
    A simple example class. Attributes match the database
       table `Sushi` exactly: id, name, price
    
    Modifying this class to match a different entity is easy:
    1. Change the file name
    2. Change the class name. Suggest something that ends in ___Model,
       like VehicleModel or PersonModel.
    3. Update the constructor (the `__init__` method) with your desired
       default values.
    4. Update the SQL statements in the `create()`, `update()`, `delete()`,
       `fetch_by_id()`, and `fetch_all()` methods. Only the SQL statements and
       the parameters to the `cursor.execute()` call need to be changed.
    5. Update the `__str__()` method so the class can be printed easily. This
       will help with debugging.
    """

    def __init__(self, data = {}):
        """
        Populates class fields, if available, or uses default values
        """
        if isinstance(data, sqlite3.Row):
            data = dict(data)

        # Assign the values in the dict to the instance variables
        self.id = data.get('id', None)
        self.name = data.get('name', '')
        self.price = data.get('price', 0.0)

    ####################
    # Method definitions
    # These require the instance variables already be populated
    
    def create(self):
        """ () -> sushi
        Creates a new sushi record in the databse, and returns the
        newly created object.
        No id is passed, as that's assigned by the database
        """
        cursor = DbHelper.get_cursor()
        cursor.execute(
            'INSERT INTO sushi(name, price) VALUES (?,?)',
            (self.name, self.price)
        )
        self.id = cursor.lastrowid
        return self
    
    def update(self):
        cursor = DbHelper.get_cursor()
        cursor.execute(
            'UPDATE sushi SET name=?, price=? WHERE id=?',
            (self.name, self.price, self.id)
        )
        return self
    
    def delete(self):
        cursor = DbHelper.get_cursor()
        cursor.execute(
            'DELETE FROM sushi WHERE id = ?',
            (self.id,) # need the trailing comma
        )
        return None



    ####################
    # Static methods
    # These don't require an instance already exist
    
    @staticmethod
    def fetch_by_id(find_me):
        cursor = DbHelper.get_cursor()
        cursor.execute(
            'SELECT id, name, price FROM sushi WHERE id = ?',
            (find_me,) # need the trailing comma
        )
        s = cursor.fetchone()
        return SushiModel(s) if s is not None else None

    @staticmethod
    def fetch_all():
        """ () -> [SushiModel, ...]
        Fetches all sushi from the database, and returns as a list
        of SushiModel objects.
        """
        cur = DbHelper.get_cursor()
        cur.execute('SELECT id, name, price FROM sushi')
        result = [SushiModel(row) for row in cur.fetchall()]
        return result
        
    # Override what an instance looks like when printed
    def __str__(self):
        """ This method allows you to print instances """
        return f'{self.id}: {self.name}, {self.price}'
